from mongoengine import Document, StringField, DateTimeField, IntField, DictField
from datetime import datetime
import uuid


class CaseDocument(Document):
    """Document model for case-related documents"""
    meta = {'collection': 'case_documents'}

    # Primary fields
    document_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    case_id = StringField()  # Reference to Case (optional for admin uploads)
    user_id = StringField(required=True)  # Owner/uploader of the document
    uploaded_by = StringField()  # Name of uploader (for display)

    # File information
    file_name = StringField(required=True)
    original_file_name = StringField(required=True)
    file_path = StringField(required=True)  # Storage path
    file_size = IntField(required=True)  # Size in bytes
    file_type = StringField()  # MIME type (e.g., application/pdf)
    file_extension = StringField()  # e.g., pdf, docx

    # Document categorization
    document_type = StringField(
        required=True,
        choices=['question', 'answer', 'evidence', 'contract', 'report', 'other', 'general', 'product'],
        default='general'
    )
    description = StringField()  # Document description

    # Processing status
    processing_status = StringField(
        required=True,
        choices=['queued', 'processing', 'completed', 'failed'],
        default='queued'
    )
    processing_error = StringField()  # Error message if processing failed

    # AI response linkage
    ai_response_id = StringField()  # Link to AI processing result

    # Metadata
    metadata = DictField()  # Additional document metadata
    tags = StringField()  # Comma-separated tags

    # Soft delete
    is_deleted = StringField(default=False)
    deleted_at = DateTimeField()
    deleted_by = StringField()

    # Timestamps
    uploaded_at = DateTimeField(default=datetime.utcnow)
    created_at = DateTimeField(default=datetime.utcnow)
    updated_at = DateTimeField(default=datetime.utcnow)

    def __repr__(self):
        return f'<CaseDocument {self.document_id}: {self.file_name}>'

    def soft_delete(self, deleted_by_user_id):
        """Soft delete the document"""
        self.is_deleted = True
        self.deleted_at = datetime.utcnow()
        self.deleted_by = deleted_by_user_id
        self.save()

    def restore(self):
        """Restore a soft-deleted document"""
        self.is_deleted = False
        self.deleted_at = None
        self.deleted_by = None
        self.save()

    @staticmethod
    def get_file_extension(filename):
        """Extract file extension from filename"""
        if '.' in filename:
            return filename.rsplit('.', 1)[1].lower()
        return ''

    @staticmethod
    def format_file_size(size_bytes):
        """Format file size in human-readable format"""
        for unit in ['B', 'KB', 'MB', 'GB']:
            if size_bytes < 1024.0:
                return f"{size_bytes:.2f} {unit}"
            size_bytes /= 1024.0
        return f"{size_bytes:.2f} TB"
