from mongoengine import Document, StringField, BooleanField, DateTimeField
from datetime import datetime, timedelta
import uuid
import secrets


class RefreshToken(Document):
    """Model for storing refresh tokens for JWT authentication"""

    meta = {'collection': 'refresh_tokens'}

    # Primary fields
    token_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    user_id = StringField(required=True)
    refresh_token = StringField(required=True, unique=True)

    # Device tracking
    device_type = StringField(default=None)
    device_fingerprint = StringField(default=None)
    mac_id = StringField(default=None)
    location = StringField(default=None)

    # Status fields
    is_used = BooleanField(default=False)
    is_expired = BooleanField(default=False)
    is_revoked = BooleanField(default=False)

    # Timestamps
    expires_at = DateTimeField(required=True)
    created_at = DateTimeField(default=datetime.utcnow)
    used_at = DateTimeField(default=None)
    revoked_at = DateTimeField(default=None)

    # Metadata
    user_agent = StringField(default=None)
    ip_address = StringField(default=None)

    def is_valid(self):
        """Check if refresh token is still valid"""
        if self.is_used or self.is_expired or self.is_revoked:
            return False

        # Check expiration
        if datetime.utcnow() > self.expires_at:
            self.is_expired = True
            self.save()
            return False

        return True

    def mark_as_used(self):
        """Mark token as used"""
        self.is_used = True
        self.used_at = datetime.utcnow()
        self.save()

    def revoke(self):
        """Revoke the token"""
        self.is_revoked = True
        self.revoked_at = datetime.utcnow()
        self.save()

    @staticmethod
    def generate_refresh_token():
        """Generate a secure refresh token"""
        return secrets.token_urlsafe(64)

    @staticmethod
    def create_refresh_token(user_id, device_type=None, device_fingerprint=None,
                           mac_id=None, location=None, user_agent=None,
                           ip_address=None, expires_in_days=30):
        """
        Create a new refresh token for a user

        Args:
            user_id: User ID to associate with token
            device_type: Type of device (mobile, web, desktop, etc.)
            device_fingerprint: Unique device identifier
            mac_id: MAC address of device
            location: Location data (JSON string or dict)
            user_agent: User agent string
            ip_address: IP address of request
            expires_in_days: Number of days until token expires (default: 30)

        Returns:
            RefreshToken: Created refresh token instance
        """
        import json

        # Convert location dict to JSON string if needed
        if isinstance(location, dict):
            location = json.dumps(location)

        token = RefreshToken(
            user_id=user_id,
            refresh_token=RefreshToken.generate_refresh_token(),
            device_type=device_type,
            device_fingerprint=device_fingerprint,
            mac_id=mac_id,
            location=location,
            user_agent=user_agent,
            ip_address=ip_address,
            expires_at=datetime.utcnow() + timedelta(days=expires_in_days)
        )
        token.save()
        return token

    @staticmethod
    def revoke_all_user_tokens(user_id):
        """Revoke all refresh tokens for a user"""
        tokens = RefreshToken.objects(user_id=user_id, is_revoked=False)
        for token in tokens:
            token.revoke()

    @staticmethod
    def cleanup_expired_tokens():
        """Remove expired tokens from database (maintenance task)"""
        expired_tokens = RefreshToken.objects(expires_at__lt=datetime.utcnow())
        count = expired_tokens.count()
        expired_tokens.delete()
        return count
